import { useState } from 'react';
import styles from './VendorStyles';
import { Text, View, TouchableWithoutFeedback, I18nManager } from 'react-native';
import Animated, {
    withSpring,
    useAnimatedStyle,
    useSharedValue,
} from 'react-native-reanimated';
import ScrollView from '../../src/components/ScrollView/ScrollView';
import BackNavigation from '../Utilities/CustomHeader/BackNavigation';
import VendorProfile from '../../src/components/Vendor/VendorProfile/VendorProfile';
import AllProducts from '../../src/components/Vendor/AllProducts/AllProducts';
import ProfileCard from '../../src/components/Vendor/ProfileCard/ProfileCard';
import ProgressiveImage from '../../src/components/ProgressiveImage';
import { useDispatch } from 'react-redux';
import { resetVendorProfile } from '../../redux/slices/vendor/vendorProfile';
import useLangTranslation from '../../hooks/useLangTranslation';

const Vendor = (props) => {
    const { trans } = useLangTranslation();
    const dispatch = useDispatch();
    const { shop_alias, vendor } = props.route.params;
    const [tab, setTab] = useState('All Products');
    const [indicatorWidth, setIndicatorWidth] = useState(0);
    const offset = useSharedValue(0);
    const customSpringStyles = useAnimatedStyle(() => {
        return {
            transform: [
                {
                    translateX: withSpring(offset.value, {
                        damping: 20,
                        stiffness: 90,
                    }),
                },
            ],
        };
    });

    return (
        <>
            <BackNavigation
                navigationProps={props.navigation}
                routeName={trans('Vendor')}
                capitalize={true}
            />
            <ScrollView>
                <View style={styles.container}>
                    {Object.keys(vendor)?.length > 0 && <ProfileCard vendor={vendor} />}
                    <View style={styles.banner}>
                        <ProgressiveImage
                            source={{ uri: vendor?.cover }}
                            style={styles.bannerImg}
                        />
                    </View>
                    <View>
                        <View>
                            <Animated.View style={customSpringStyles}>
                                {tab === 'All Products' && (
                                    <Indicator
                                        setIndicatorWidth={setIndicatorWidth}
                                    />
                                )}
                                {tab === 'Vendor Profiles' && (
                                    <Indicator
                                        setIndicatorWidth={setIndicatorWidth}
                                    />
                                )}
                            </Animated.View>
                            <View style={styles.slideContainer}>
                                <TouchableWithoutFeedback
                                    onPress={() => {
                                        setTab('All Products');
                                        offset.value =
                                            indicatorWidth - indicatorWidth;
                                    }}
                                >
                                    <Text
                                        style={[
                                            styles.slideItem,
                                            styles.color(tab, 'All Products')
                                        ]}
                                    >
                                        {trans('All Products')}
                                    </Text>
                                </TouchableWithoutFeedback>
                                <TouchableWithoutFeedback
                                    onPress={() => {
                                        offset.value = indicatorWidth;
                                        dispatch(resetVendorProfile());
                                        setTab('Vendor Profiles');
                                    }}
                                >
                                    <Text
                                        style={[
                                            styles.slideItem,
                                            styles.color(tab, 'Vendor Profiles')
                                        ]}
                                    >
                                        {trans('Vendor Profile')}
                                    </Text>
                                </TouchableWithoutFeedback>
                            </View>
                        </View>
                        <View style={styles.slideScreen}>
                            {tab === 'All Products' && (
                                <AllProducts shop_alias={shop_alias} />
                            )}
                            {tab === 'Vendor Profiles' && (
                                <VendorProfile shop_alias={shop_alias} vendor={vendor} />
                            )}
                        </View>
                    </View>
                </View>
            </ScrollView>
        </>
    );
};

export default Vendor;

function Indicator({ setIndicatorWidth }) {
    return (
        <View
            style={styles.indicator}
            onLayout={(event) =>
                setIndicatorWidth(I18nManager.isRTL 
                    ? -event.nativeEvent.layout.width 
                    : event.nativeEvent.layout.width
                )
            }
        />
    );
}
